const API_LIST = 'https://kmlgekkbhvapegniuapw.supabase.co/functions/v1/api-list';
const API_SET  = 'https://kmlgekkbhvapegniuapw.supabase.co/functions/v1/api-set';

const ALARM_NAME = 'tracking_poll_notifications';
const POLLING_MINUTES = 1; // 60s

/* =========================
   SESSION
========================= */
async function getSessionToken() {
    const { sessionToken } = await chrome.storage.local.get('sessionToken');
    return sessionToken || null;
}

/* =========================
   API LIST
========================= */
async function fetchNotifications() {
    const token = await getSessionToken();

    if (!token) {
        console.log('[Tracking BG] ❌ Sem sessionToken');
        return [];
    }

    const url = `${API_LIST}?session=${encodeURIComponent(token)}`;
    console.log('[Tracking BG] 🌐 GET', url);

    try {
        const res = await fetch(url, { cache: 'no-store' });
        const data = await res.json();

        console.log('[Tracking BG] 📦 Response:', data);

        if (!res.ok) return [];

        if (Array.isArray(data.notifications)) return data.notifications;
        if (Array.isArray(data)) return data;

        return [];
    } catch (e) {
        console.error('[Tracking BG] 💥 Fetch error:', e);
        return [];
    }
}

/* =========================
   API SET (MARK AS READ)
========================= */
async function markAsRead(notificationId) {
    const token = await getSessionToken();
    if (!token || !notificationId) return;

    const url =
        `${API_SET}?session=${encodeURIComponent(token)}` +
        `&notification=${encodeURIComponent(notificationId)}`;

    console.log('[Tracking BG] 📝 Mark as read:', url);

    try {
        await fetch(url, { method: 'GET' });
    } catch (e) {
        console.error('[Tracking BG] 💥 Mark error:', e);
    }
}

/* =========================
   POLL
========================= */
async function pollNotifications() {
    const notifications = await fetchNotifications();
    const now = Date.now();

    console.log('[Tracking BG] 🔔 Pending:', notifications);

    chrome.storage.local.set({
        pendingNotifications: notifications,
        lastPoll: now
    });
}

/* =========================
   ALARM SETUP
========================= */
function setupAlarm() {
    chrome.alarms.clear(ALARM_NAME, () => {
        chrome.alarms.create(ALARM_NAME, {
            periodInMinutes: POLLING_MINUTES
        });
    });
}

/* =========================
   LIFECYCLE
========================= */
chrome.runtime.onInstalled.addListener(() => {
    console.log('[Tracking BG] 📦 Installed');
    setupAlarm();
    pollNotifications();
});

chrome.runtime.onStartup.addListener(() => {
    console.log('[Tracking BG] 🌅 Startup');
    setupAlarm();
    pollNotifications();
});

chrome.alarms.onAlarm.addListener(alarm => {
    if (alarm.name === ALARM_NAME) {
        console.log('[Tracking BG] ⏰ Alarm fired');
        pollNotifications();
    }
});

/* =========================
   FORCE POLL ON TAB EVENTS
========================= */
chrome.tabs.onActivated.addListener(() => {
    pollNotifications();
});

chrome.tabs.onUpdated.addListener((tabId, info) => {
    if (info.status === 'complete') {
        pollNotifications();
    }
});

/* =========================
   MESSAGES FROM UI
========================= */
chrome.runtime.onMessage.addListener((message) => {
    if (message?.type === 'MARK_AS_READ') {
        markAsRead(message.notificationId);
    }
});
