(() => {
    'use strict';

    // Gerar ID único para esta instância
    const UNIQUE_ID = 'tn-' + Math.random().toString(36).substr(2, 9) + '-' + Date.now();
    const ROOT_ID = `tracking-ui-root-${UNIQUE_ID}`;
    const CENTER_VISIBLE_MS = 1000;
    const BASE_AUTO_HIDE_MS = 10000;
    const AUTO_HIDE_INCREMENT_MS = 2000;
    const POLL_MS = 30000;
    const DOCK_AUTO_HIDE_MS = 10000;

    // Classes únicas e aleatórias
    const CLS = {
        root: `tn-root-${UNIQUE_ID}`,
        backdrop: `tn-backdrop-${UNIQUE_ID}`,
        particles: `tn-particles-${UNIQUE_ID}`,
        particle: `tn-particle-${UNIQUE_ID}`,
        dock: `tn-dock-${UNIQUE_ID}`,
        card: `tn-card-${UNIQUE_ID}`,
        floating: `tn-floating-${UNIQUE_ID}`,
        dockItem: `tn-dockitem-${UNIQUE_ID}`,
        icon: `tn-icon-${UNIQUE_ID}`,
        iconEmoji: `tn-iconemoji-${UNIQUE_ID}`,
        content: `tn-content-${UNIQUE_ID}`,
        type: `tn-type-${UNIQUE_ID}`,
        title: `tn-title-${UNIQUE_ID}`,
        text: `tn-text-${UNIQUE_ID}`,
        actions: `tn-actions-${UNIQUE_ID}`,
        btnRead: `tn-btnread-${UNIQUE_ID}`,
        btnClose: `tn-btnclose-${UNIQUE_ID}`,
        enter: `tn-enter-${UNIQUE_ID}`,
        exit: `tn-exit-${UNIQUE_ID}`,
        show: `tn-show-${UNIQUE_ID}`,
        visible: `tn-visible-${UNIQUE_ID}`
    };

    // FLAGS DE CONFIGURAÇÃO
    const BUSINESS_HOURS_ONLY = true;
    const BUSINESS_START_HOUR = 9;
    const BUSINESS_END_HOUR = 18;

    const TYPE_META = {
        notificacao: { label: 'Notificação', color: '#2563eb', icon: '🔔' },
        evento:       { label: 'Evento',       color: '#7c3aed', icon: '📅' },
        launch:       { label: 'Lançamento',    color: '#db2777', icon: '🚀' },
        update:       { label: 'Atualização',   color: '#059669', icon: '✨' }
    };

    const renderedFloating = new Set();
    const renderedDock = new Set();
    let running = false;
    let pollInterval = null;

    /* =========================
       EXTENSION CONTEXT CHECK
    ========================= */
    function isExtensionValid() {
        try {
            return chrome.runtime && chrome.runtime.id;
        } catch (e) {
            return false;
        }
    }

    function cleanup() {
        if (pollInterval) {
            clearInterval(pollInterval);
            pollInterval = null;
        }
        const root = document.getElementById(ROOT_ID);
        if (root) root.remove();
        console.log('[Tracking UI] Extension context invalidated - cleaning up');
    }

    /* =========================
       BUSINESS HOURS CHECK
    ========================= */
    function isBusinessHours() {
        if (!BUSINESS_HOURS_ONLY) return true;
        const now = new Date();
        const hour = now.getHours();
        return hour >= BUSINESS_START_HOUR && hour < BUSINESS_END_HOUR;
    }

    /* =========================
       ROOT
    ========================= */
    function ensureRoot() {
        let root = document.getElementById(ROOT_ID);
        if (root) return root;

        root = document.createElement('div');
        root.id = ROOT_ID;
        root.className = CLS.root;
        root.innerHTML = `
            <div class="${CLS.backdrop}"></div>
            <div class="${CLS.particles}"></div>
            <div class="${CLS.dock}"></div>
        `;
        document.documentElement.appendChild(root);
        return root;
    }

    /* =========================
       PARTICLES EFFECT
    ========================= */
    function createParticles(x, y, color) {
        const root = document.getElementById(ROOT_ID);
        if (!root) return;
        const container = root.querySelector(`.${CLS.particles}`);
        if (!container) return;

        for (let i = 0; i < 12; i++) {
            const particle = document.createElement('div');
            particle.className = CLS.particle;
            particle.style.left = x + 'px';
            particle.style.top = y + 'px';
            particle.style.background = color;

            const angle = (i / 12) * Math.PI * 2;
            const velocity = 80 + Math.random() * 40;
            const tx = Math.cos(angle) * velocity;
            const ty = Math.sin(angle) * velocity;

            particle.style.setProperty('--tx', tx + 'px');
            particle.style.setProperty('--ty', ty + 'px');

            container.appendChild(particle);

            setTimeout(() => {
                if (particle.parentElement) particle.remove();
            }, 1000);
        }
    }

    /* =========================
       CARD
    ========================= */
    function buildCard(notif, isDock = false) {
        const type = TYPE_META[notif.categoria] || TYPE_META.notificacao;

        const el = document.createElement('div');
        el.className = CLS.card;
        el.dataset.id = notif.id;

        el.innerHTML = `
            <div class="${CLS.icon}" style="background: linear-gradient(135deg, ${type.color}15, ${type.color}05);">
                <!-- SUBSTITUIR ESTE COMENTÁRIO PELO SVG DA HIGHEND -->
                <span class="${CLS.iconEmoji}">${type.icon}</span>
            </div>
            <div class="${CLS.content}">
                <div class="${CLS.type}" style="color:${type.color}">
                    ${type.label}
                </div>
                <div class="${CLS.title}">${notif.titulo}</div>
                ${notif.texto ? `<div class="${CLS.text}">${notif.texto}</div>` : ''}
            </div>
            <div class="${CLS.actions}">
                <button class="${CLS.btnRead}" title="Marcar como lida">✓</button>
                <button class="${CLS.btnClose}" title="Fechar">✕</button>
            </div>
        `;

        const readBtn = el.querySelector(`.${CLS.btnRead}`);
        const closeBtn = el.querySelector(`.${CLS.btnClose}`);

        readBtn.onclick = (e) => {
            e.stopPropagation();
            if (!isExtensionValid()) {
                cleanup();
                return;
            }

            const rect = readBtn.getBoundingClientRect();
            createParticles(rect.left + rect.width / 2, rect.top + rect.height / 2, type.color);

            try {
                chrome.runtime.sendMessage({
                    type: 'MARK_AS_READ',
                    notificationId: notif.id
                }, response => {
                    if (chrome.runtime.lastError) {
                        console.log('[Tracking UI] Extension context lost');
                        cleanup();
                    }
                });
            } catch (e) {
                cleanup();
            }

            removeCard(el, isDock);
        };

        closeBtn.onclick = (e) => {
            e.stopPropagation();
            removeCard(el, isDock);
        };

        return el;
    }

    /* =========================
       PRESENTATION (ENHANCED)
    ========================= */
    async function present(list) {
        if (running || !isExtensionValid()) return;
        running = true;

        const root = ensureRoot();
        const backdrop = root.querySelector(`.${CLS.backdrop}`);
        const dock = root.querySelector(`.${CLS.dock}`);

        if (!backdrop || !dock) {
            running = false;
            return;
        }

        backdrop.classList.add(CLS.visible);

        const cardsToShow = [];

        for (const notif of list) {
            if (renderedFloating.has(notif.id)) continue;
            renderedFloating.add(notif.id);
            cardsToShow.push(notif);
        }

        // Mostrar todas as notificações sequencialmente com animação
        for (let i = 0; i < cardsToShow.length; i++) {
            if (!isExtensionValid()) {
                cleanup();
                return;
            }

            const notif = cardsToShow[i];
            const type = TYPE_META[notif.categoria] || TYPE_META.notificacao;

            const card = buildCard(notif, false);
            card.classList.add(CLS.floating);
            card.style.setProperty('--tn-card-color', type.color);
            root.appendChild(card);

            // Delay entre notificações para efeito sequencial
            await delay(i * 300);

            // Animação de entrada suave
            await nextFrame();
            card.classList.add(CLS.enter);

            // Aguardar visualização
            await delay(CENTER_VISIBLE_MS);

            // Iniciar saída
            card.classList.add(CLS.exit);

            // Adicionar ao dock antes de remover completamente
            setTimeout(() => {
                if (card.parentElement) card.remove();
                if (isExtensionValid()) {
                    addToDock(notif, dock);
                }
            }, 600);
        }

        backdrop.classList.remove(CLS.visible);
        running = false;
    }

    function addToDock(notif, dock) {
        if (renderedDock.has(notif.id) || !isExtensionValid()) return;
        renderedDock.add(notif.id);

        const dockCard = buildCard(notif, true);
        dockCard.classList.add(CLS.dockItem);
        dock.appendChild(dockCard);

        requestAnimationFrame(() => {
            if (dockCard.parentElement) {
                dockCard.classList.add(CLS.show);
            }
        });

        // Auto-hide após 10s
        setTimeout(() => {
            if (dockCard.parentElement && isExtensionValid()) {
                removeCard(dockCard, true);
            }
        }, DOCK_AUTO_HIDE_MS);
    }

    function removeCard(card, isDock = false) {
        const notifId = card.dataset.id;

        card.classList.add(CLS.exit);
        setTimeout(() => {
            if (card.parentElement) card.remove();
            if (isDock) {
                renderedDock.delete(notifId);
            } else {
                renderedFloating.delete(notifId);
            }
        }, 400);
    }

    /* =========================
       STORAGE
    ========================= */
    function evaluate() {
        if (!isBusinessHours() || !isExtensionValid()) return;

        try {
            chrome.storage.local.get(['pendingNotifications'], r => {
                if (chrome.runtime.lastError) {
                    console.log('[Tracking UI] Storage error:', chrome.runtime.lastError);
                    cleanup();
                    return;
                }
                const list = r.pendingNotifications || [];
                if (list.length) present(list);
            });
        } catch (e) {
            console.log('[Tracking UI] Extension context lost during storage access');
            cleanup();
        }
    }

    // Initial evaluation
    if (isExtensionValid()) {
        evaluate();

        pollInterval = setInterval(() => {
            if (!isExtensionValid()) {
                cleanup();
                return;
            }
            if (isBusinessHours()) {
                evaluate();
            }
        }, POLL_MS);

        try {
            chrome.storage.onChanged.addListener((c, a) => {
                if (!isExtensionValid()) {
                    cleanup();
                    return;
                }
                if (a === 'local' && c.pendingNotifications && isBusinessHours()) {
                    evaluate();
                }
            });
        } catch (e) {
            console.log('[Tracking UI] Failed to add storage listener');
            cleanup();
        }
    }

    /* =========================
       UTILS
    ========================= */
    const delay = ms => new Promise(r => setTimeout(r, ms));
    const nextFrame = () => new Promise(r => requestAnimationFrame(r));

    /* =========================
       STYLES (ISOLATED)
    ========================= */
    const style = document.createElement('style');
    style.textContent = `
        /* RESET COMPLETO PARA ISOLAMENTO */
        #${ROOT_ID},
        #${ROOT_ID} *,
        #${ROOT_ID} *::before,
        #${ROOT_ID} *::after {
            all: initial;
            box-sizing: border-box;
            font-family: -apple-system, BlinkMacSystemFont, 'SF Pro Display', 'Segoe UI', sans-serif;
        }

        #${ROOT_ID} {
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            right: 0 !important;
            bottom: 0 !important;
            z-index: 2147483647 !important;
            pointer-events: none !important;
            margin: 0 !important;
            padding: 0 !important;
            border: none !important;
            outline: none !important;
            background: none !important;
        }

        .${CLS.backdrop} {
            position: absolute !important;
            top: 0 !important;
            left: 0 !important;
            right: 0 !important;
            bottom: 0 !important;
            background: rgba(0, 0, 0, 0.5) !important;
            backdrop-filter: blur(12px) saturate(120%) !important;
            -webkit-backdrop-filter: blur(12px) saturate(120%) !important;
            opacity: 0 !important;
            transition: opacity 0.5s cubic-bezier(0.4, 0, 0.2, 1) !important;
            pointer-events: none !important;
        }
        
        .${CLS.backdrop}.${CLS.visible} {
            opacity: 1 !important;
        }

        .${CLS.particles} {
            position: absolute !important;
            top: 0 !important;
            left: 0 !important;
            right: 0 !important;
            bottom: 0 !important;
            pointer-events: none !important;
            overflow: hidden !important;
        }

        .${CLS.particle} {
            position: absolute !important;
            width: 6px !important;
            height: 6px !important;
            border-radius: 50% !important;
            pointer-events: none !important;
            animation: tn-particle-burst-${UNIQUE_ID} 0.8s cubic-bezier(0.4, 0, 0.2, 1) forwards !important;
            box-shadow: 0 0 8px currentColor !important;
        }

        @keyframes tn-particle-burst-${UNIQUE_ID} {
            0% {
                transform: translate(0, 0) scale(1);
                opacity: 1;
            }
            100% {
                transform: translate(var(--tx), var(--ty)) scale(0);
                opacity: 0;
            }
        }

        .${CLS.card} {
            background: #F5F6FA !important;
            border-radius: 16px !important;
            box-shadow: 
                0 4px 16px rgba(0, 0, 0, 0.08),
                0 1px 4px rgba(0, 0, 0, 0.04) !important;
            padding: 16px !important;
            display: grid !important;
            grid-template-columns: auto 1fr auto !important;
            gap: 14px !important;
            width: 360px !important;
            max-width: 90vw !important;
            pointer-events: auto !important;
            position: relative !important;
            overflow: hidden !important;
            cursor: default !important;
            margin: 0 !important;
            border: none !important;
        }

        .${CLS.card}::before {
            content: '' !important;
            position: absolute !important;
            top: 0 !important;
            left: 0 !important;
            right: 0 !important;
            height: 3px !important;
            background: linear-gradient(90deg, var(--tn-card-color), transparent) !important;
            opacity: 0.8 !important;
        }

        /* FLOATING */
        .${CLS.card}.${CLS.floating} {
            position: absolute !important;
            left: 50% !important;
            bottom: -200px !important;
            transform: translateX(-50%) translateY(0) scale(0.88) !important;
            opacity: 0 !important;
            filter: blur(10px) !important;
            transition:
                transform 0.75s cubic-bezier(0.34, 1.56, 0.64, 1),
                opacity 0.6s cubic-bezier(0.4, 0, 0.2, 1),
                filter 0.6s cubic-bezier(0.4, 0, 0.2, 1),
                bottom 0.75s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
        }

        .${CLS.card}.${CLS.floating}.${CLS.enter} {
            bottom: 50% !important;
            transform: translate(-50%, 50%) scale(1) !important;
            opacity: 1 !important;
            filter: blur(0) !important;
        }

        .${CLS.card}.${CLS.floating}.${CLS.exit} {
            opacity: 0 !important;
            transform: translate(-50%, -100%) scale(0.92) !important;
            filter: blur(8px) !important;
            transition:
                transform 0.6s cubic-bezier(0.4, 0, 1, 1),
                opacity 0.5s ease,
                filter 0.5s ease !important;
        }

        /* DOCK */
        .${CLS.dock} {
            position: absolute !important;
            bottom: 24px !important;
            left: 24px !important;
            display: flex !important;
            flex-direction: column-reverse !important;
            gap: 10px !important;
            max-height: calc(100vh - 48px) !important;
            overflow-y: auto !important;
            padding: 8px !important;
            margin: -8px !important;
        }

        .${CLS.dock}::-webkit-scrollbar {
            width: 6px !important;
        }

        .${CLS.dock}::-webkit-scrollbar-track {
            background: transparent !important;
        }

        .${CLS.dock}::-webkit-scrollbar-thumb {
            background: rgba(0, 0, 0, 0.15) !important;
            border-radius: 3px !important;
        }

        .${CLS.card}.${CLS.dockItem} {
            width: 340px !important;
            opacity: 0 !important;
            transform: translateX(-60px) scale(0.9) !important;
            transition: all 0.45s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
        }

        .${CLS.card}.${CLS.dockItem}.${CLS.show} {
            opacity: 1 !important;
            transform: translateX(0) scale(1) !important;
        }

        .${CLS.card}.${CLS.dockItem}.${CLS.exit} {
            opacity: 0 !important;
            transform: translateX(-60px) scale(0.9) !important;
            transition: all 0.3s ease !important;
        }

        .${CLS.card}.${CLS.dockItem}:hover {
            transform: translateX(4px) scale(1.02) !important;
            box-shadow: 
                0 8px 24px rgba(0, 0, 0, 0.12),
                0 2px 8px rgba(0, 0, 0, 0.06) !important;
        }

        .${CLS.icon} {
            width: 44px !important;
            height: 44px !important;
            border-radius: 12px !important;
            background: rgba(15, 23, 42, 0.04) !important;
            flex-shrink: 0 !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            align-self: start !important;
            transition: transform 0.2s ease !important;
            margin: 0 !important;
            padding: 0 !important;
        }

        .${CLS.card}:hover .${CLS.icon} {
            transform: scale(1.08) rotate(3deg) !important;
        }

        .${CLS.iconEmoji} {
            font-size: 22px !important;
            line-height: 1 !important;
            filter: drop-shadow(0 2px 4px rgba(0, 0, 0, 0.08)) !important;
            display: block !important;
        }

        .${CLS.content} {
            display: flex !important;
            flex-direction: column !important;
            gap: 5px !important;
            min-width: 0 !important;
            align-self: center !important;
            margin: 0 !important;
            padding: 0 !important;
        }

        .${CLS.type} {
            font-size: 10px !important;
            font-weight: 700 !important;
            text-transform: uppercase !important;
            letter-spacing: 0.8px !important;
            opacity: 0.85 !important;
            margin: 0 !important;
            padding: 0 !important;
            line-height: 1.2 !important;
        }

        .${CLS.title} {
            font-size: 15px !important;
            font-weight: 600 !important;
            color: #0f172a !important;
            line-height: 1.3 !important;
            word-wrap: break-word !important;
            overflow-wrap: break-word !important;
            margin: 0 !important;
            padding: 0 !important;
        }

        .${CLS.text} {
            font-size: 13px !important;
            color: #475569 !important;
            line-height: 1.4 !important;
            margin: 0 !important;
            padding: 0 !important;
            word-wrap: break-word !important;
            overflow-wrap: break-word !important;
        }

        .${CLS.actions} {
            display: flex !important;
            flex-direction: column !important;
            gap: 6px !important;
            align-self: center !important;
            margin: 0 !important;
            padding: 0 !important;
        }

        .${CLS.btnRead},
        .${CLS.btnClose} {
            width: 32px !important;
            height: 32px !important;
            border-radius: 10px !important;
            border: none !important;
            background: rgba(15, 23, 42, 0.06) !important;
            cursor: pointer !important;
            font-size: 14px !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1) !important;
            color: #475569 !important;
            margin: 0 !important;
            padding: 0 !important;
            outline: none !important;
            line-height: 1 !important;
        }

        .${CLS.btnRead}:hover,
        .${CLS.btnClose}:hover {
            background: rgba(15, 23, 42, 0.12) !important;
            transform: scale(1.1) !important;
        }

        .${CLS.btnRead}:active,
        .${CLS.btnClose}:active {
            transform: scale(0.95) !important;
        }

        .${CLS.btnRead}:hover {
            background: #10b98120 !important;
            color: #10b981 !important;
        }

        .${CLS.btnClose}:hover {
            background: #ef444420 !important;
            color: #ef4444 !important;
        }

        @media (max-width: 640px) {
            .${CLS.card} {
                width: calc(100vw - 48px) !important;
                padding: 14px !important;
            }

            .${CLS.dock} {
                left: 16px !important;
                right: 16px !important;
                bottom: 16px !important;
            }

            .${CLS.card}.${CLS.dockItem} {
                width: calc(100vw - 32px) !important;
            }
        }

        @media (prefers-reduced-motion: reduce) {
            .${CLS.card}.${CLS.floating},
            .${CLS.card}.${CLS.dockItem},
            .${CLS.backdrop},
            .${CLS.btnRead},
            .${CLS.btnClose},
            .${CLS.icon} {
                transition-duration: 0.01ms !important;
            }
        }
    `;
    document.head.appendChild(style);
})();